import React, { useState } from 'react';
import {
  View,
  Text,
  FlatList,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Platform,
} from 'react-native';
import { Ionicons, MaterialIcons } from '@expo/vector-icons';
import { useSafeAreaInsets } from 'react-native-safe-area-context';
import { useRouter } from 'expo-router';
import { useTheme } from '../../hooks/useTheme';
import { usePosts } from '../../hooks/usePosts';
import { useStories } from '../../hooks/useStories';
import { useResponsive } from '../../hooks/useResponsive';
import { PostCard } from '../../components/PostCard';
import { StoryAvatar } from '../../components/StoryAvatar';
import { CommentsModal } from '../../components/CommentsModal';
import { CustomAlert } from '../../components/ui/CustomAlert';
import { SPACING, TYPOGRAPHY, LAYOUT } from '../../constants/theme';

export default function HomeScreen() {
  const { colors } = useTheme();
  const { posts } = usePosts();
  const { stories } = useStories();
  const { isDesktop } = useResponsive();
  const insets = useSafeAreaInsets();
  const router = useRouter();

  const [selectedPostId, setSelectedPostId] = useState<string | null>(null);
  const [shareAlert, setShareAlert] = useState({
    visible: false,
    title: '',
    message: '',
  });

  const handleCommentPress = (postId: string) => {
    setSelectedPostId(postId);
  };

  const handleSharePress = () => {
    setShareAlert({
      visible: true,
      title: 'Share Post',
      message: 'Sharing functionality will be implemented soon!',
    });
  };

  const renderHeader = () => (
    <>
      <View
        style={[
          styles.header,
          {
            backgroundColor: colors.background,
            borderBottomColor: colors.border,
            paddingTop: insets.top,
          },
        ]}
      >
        <Text style={[styles.logo, { color: colors.text }]}>Demuregram</Text>
        <View style={styles.headerIcons}>
          <TouchableOpacity style={styles.headerIcon}>
            <Ionicons name="heart-outline" size={26} color={colors.text} />
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.headerIcon}
            onPress={() => router.push('/settings')}
          >
            <Ionicons name="paper-plane-outline" size={26} color={colors.text} />
          </TouchableOpacity>
        </View>
      </View>

      <View style={[styles.storiesContainer, { borderBottomColor: colors.border }]}>
        <ScrollView horizontal showsHorizontalScrollIndicator={false}>
          <View style={styles.storiesContent}>
            {stories.map(story => (
              <StoryAvatar
                key={story.id}
                avatar={story.user.avatar}
                username={story.user.username}
                isViewed={story.isViewed}
                onPress={() => {}}
              />
            ))}
          </View>
        </ScrollView>
      </View>
    </>
  );

  const renderPost = ({ item }: { item: typeof posts[0] }) => (
    <PostCard
      post={item}
      onCommentPress={() => handleCommentPress(item.id)}
      onSharePress={handleSharePress}
    />
  );

  const content = (
    <View style={[styles.container, { backgroundColor: colors.background }]}>
      <FlatList
        data={posts}
        renderItem={renderPost}
        keyExtractor={item => item.id}
        ListHeaderComponent={renderHeader}
        showsVerticalScrollIndicator={false}
        contentContainerStyle={styles.feedContent}
      />

      <CommentsModal
        visible={!!selectedPostId}
        postId={selectedPostId || ''}
        onClose={() => setSelectedPostId(null)}
      />

      <CustomAlert
        visible={shareAlert.visible}
        title={shareAlert.title}
        message={shareAlert.message}
        onDismiss={() => setShareAlert(prev => ({ ...prev, visible: false }))}
      />
    </View>
  );

  if (isDesktop) {
    return (
      <View style={styles.desktopLayout}>
        <View style={[styles.desktopSidebar, { backgroundColor: colors.background, borderRightColor: colors.border }]}>
          <Text style={[styles.desktopLogo, { color: colors.text }]}>Demuregram</Text>
          <TouchableOpacity style={styles.sidebarItem}>
            <Ionicons name="home" size={26} color={colors.text} />
            <Text style={[styles.sidebarText, { color: colors.text }]}>Home</Text>
          </TouchableOpacity>
          <TouchableOpacity style={styles.sidebarItem}>
            <Ionicons name="search" size={26} color={colors.text} />
            <Text style={[styles.sidebarText, { color: colors.text }]}>Search</Text>
          </TouchableOpacity>
          <TouchableOpacity style={styles.sidebarItem}>
            <MaterialIcons name="video-library" size={26} color={colors.text} />
            <Text style={[styles.sidebarText, { color: colors.text }]}>Clips</Text>
          </TouchableOpacity>
          <TouchableOpacity style={styles.sidebarItem} onPress={() => router.push('/creator-dashboard')}>
            <MaterialIcons name="dashboard" size={26} color={colors.text} />
            <Text style={[styles.sidebarText, { color: colors.text }]}>Dashboard</Text>
          </TouchableOpacity>
          <TouchableOpacity style={styles.sidebarItem}>
            <Ionicons name="person-circle-outline" size={26} color={colors.text} />
            <Text style={[styles.sidebarText, { color: colors.text }]}>Profile</Text>
          </TouchableOpacity>
          <TouchableOpacity style={styles.sidebarItem} onPress={() => router.push('/settings')}>
            <Ionicons name="settings-outline" size={26} color={colors.text} />
            <Text style={[styles.sidebarText, { color: colors.text }]}>Settings</Text>
          </TouchableOpacity>
        </View>
        <View style={styles.desktopContent}>{content}</View>
      </View>
    );
  }

  return content;
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
  },
  header: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: SPACING.lg,
    paddingBottom: SPACING.md,
    borderBottomWidth: 1,
  },
  logo: {
    fontSize: TYPOGRAPHY.sizes.xxl,
    fontWeight: TYPOGRAPHY.weights.bold,
    fontFamily: Platform.select({ ios: 'Helvetica Neue', default: 'sans-serif' }),
  },
  headerIcons: {
    flexDirection: 'row',
  },
  headerIcon: {
    marginLeft: SPACING.lg,
  },
  storiesContainer: {
    paddingVertical: SPACING.md,
    borderBottomWidth: 1,
  },
  storiesContent: {
    paddingHorizontal: SPACING.lg,
    flexDirection: 'row',
  },
  feedContent: {
    paddingBottom: SPACING.xl,
  },
  desktopLayout: {
    flex: 1,
    flexDirection: 'row',
  },
  desktopSidebar: {
    width: LAYOUT.sidebarWidth,
    borderRightWidth: 1,
    paddingVertical: SPACING.xxl,
    paddingHorizontal: SPACING.lg,
  },
  desktopLogo: {
    fontSize: TYPOGRAPHY.sizes.xxl,
    fontWeight: TYPOGRAPHY.weights.bold,
    marginBottom: SPACING.xxl,
    paddingLeft: SPACING.md,
  },
  sidebarItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: SPACING.md,
    paddingHorizontal: SPACING.md,
    marginBottom: SPACING.xs,
    borderRadius: 8,
    gap: SPACING.lg,
  },
  sidebarText: {
    fontSize: TYPOGRAPHY.sizes.lg,
    fontWeight: TYPOGRAPHY.weights.medium,
  },
  desktopContent: {
    flex: 1,
    maxWidth: LAYOUT.maxContentWidth,
    alignSelf: 'center',
    width: '100%',
  },
});