import React from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StyleSheet,
  ScrollView,
  Dimensions,
} from 'react-native';
import { Image } from 'expo-image';
import { MaterialIcons, Ionicons } from '@expo/vector-icons';
import { useSafeAreaInsets } from 'react-native-safe-area-context';
import { useRouter } from 'expo-router';
import { useTheme } from '../../hooks/useTheme';
import { useAuth } from '../../hooks/useAuth';
import { SPACING, TYPOGRAPHY } from '../../constants/theme';

const { width: SCREEN_WIDTH } = Dimensions.get('window');
const GRID_SIZE = (SCREEN_WIDTH - SPACING.lg * 2 - 2) / 3;

export default function ProfileScreen() {
  const { colors } = useTheme();
  const { user } = useAuth();
  const insets = useSafeAreaInsets();
  const router = useRouter();

  if (!user) return null;

  return (
    <View style={[styles.container, { backgroundColor: colors.background }]}>
      <View
        style={[
          styles.header,
          {
            paddingTop: insets.top + SPACING.md,
            borderBottomColor: colors.border,
          },
        ]}
      >
        <Text style={[styles.username, { color: colors.text }]}>{user.username}</Text>
        <View style={styles.headerIcons}>
          <TouchableOpacity style={styles.headerIcon}>
            <Ionicons name="add-circle-outline" size={26} color={colors.text} />
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.headerIcon}
            onPress={() => router.push('/settings')}
          >
            <Ionicons name="menu" size={26} color={colors.text} />
          </TouchableOpacity>
        </View>
      </View>

      <ScrollView>
        <View style={styles.profileInfo}>
          <Image source={{ uri: user.avatar }} style={styles.avatar} />
          <View style={styles.stats}>
            <View style={styles.stat}>
              <Text style={[styles.statNumber, { color: colors.text }]}>
                {user.postsCount}
              </Text>
              <Text style={[styles.statLabel, { color: colors.text }]}>Posts</Text>
            </View>
            <View style={styles.stat}>
              <Text style={[styles.statNumber, { color: colors.text }]}>
                {(user.followersCount / 1000).toFixed(1)}K
              </Text>
              <Text style={[styles.statLabel, { color: colors.text }]}>Followers</Text>
            </View>
            <View style={styles.stat}>
              <Text style={[styles.statNumber, { color: colors.text }]}>
                {user.followingCount}
              </Text>
              <Text style={[styles.statLabel, { color: colors.text }]}>Following</Text>
            </View>
          </View>
        </View>

        <View style={styles.bioSection}>
          <View style={styles.nameRow}>
            <Text style={[styles.displayName, { color: colors.text }]}>
              {user.displayName}
            </Text>
            {user.isVerified && (
              <MaterialIcons
                name="verified"
                size={18}
                color={colors.verified}
                style={styles.verifiedBadge}
              />
            )}
            {user.isMonetized && (
              <Text style={[styles.monetizedBadge, { color: colors.monetized }]}>
                $
              </Text>
            )}
          </View>
          {user.bio && (
            <Text style={[styles.bio, { color: colors.text }]}>{user.bio}</Text>
          )}
        </View>

        {user.isMonetized && (
          <TouchableOpacity
            style={[styles.dashboardButton, { backgroundColor: colors.primary }]}
            onPress={() => router.push('/creator-dashboard')}
          >
            <MaterialIcons name="dashboard" size={20} color="#FFFFFF" />
            <Text style={styles.dashboardText}>Creator Dashboard</Text>
          </TouchableOpacity>
        )}

        <View style={styles.actionButtons}>
          <TouchableOpacity
            style={[styles.actionButton, { backgroundColor: colors.surface }]}
          >
            <Text style={[styles.actionButtonText, { color: colors.text }]}>
              Edit Profile
            </Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.actionButton, { backgroundColor: colors.surface }]}
          >
            <Text style={[styles.actionButtonText, { color: colors.text }]}>
              Share Profile
            </Text>
          </TouchableOpacity>
        </View>

        <View style={styles.gridPlaceholder}>
          <Ionicons name="grid-outline" size={64} color={colors.textSecondary} />
          <Text style={[styles.gridText, { color: colors.textSecondary }]}>
            Your posts will appear here
          </Text>
        </View>
      </ScrollView>
    </View>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
  },
  header: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: SPACING.lg,
    paddingBottom: SPACING.md,
    borderBottomWidth: 1,
  },
  username: {
    fontSize: TYPOGRAPHY.sizes.xl,
    fontWeight: TYPOGRAPHY.weights.semibold,
  },
  headerIcons: {
    flexDirection: 'row',
  },
  headerIcon: {
    marginLeft: SPACING.lg,
  },
  profileInfo: {
    flexDirection: 'row',
    paddingHorizontal: SPACING.lg,
    paddingVertical: SPACING.lg,
    alignItems: 'center',
  },
  avatar: {
    width: 86,
    height: 86,
    borderRadius: 43,
  },
  stats: {
    flex: 1,
    flexDirection: 'row',
    justifyContent: 'space-around',
    paddingLeft: SPACING.xl,
  },
  stat: {
    alignItems: 'center',
  },
  statNumber: {
    fontSize: TYPOGRAPHY.sizes.xl,
    fontWeight: TYPOGRAPHY.weights.semibold,
  },
  statLabel: {
    fontSize: TYPOGRAPHY.sizes.sm,
  },
  bioSection: {
    paddingHorizontal: SPACING.lg,
    marginBottom: SPACING.lg,
  },
  nameRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: SPACING.xs,
  },
  displayName: {
    fontSize: TYPOGRAPHY.sizes.base,
    fontWeight: TYPOGRAPHY.weights.semibold,
  },
  verifiedBadge: {
    marginLeft: 4,
  },
  monetizedBadge: {
    marginLeft: 4,
    fontSize: TYPOGRAPHY.sizes.lg,
    fontWeight: TYPOGRAPHY.weights.bold,
  },
  bio: {
    fontSize: TYPOGRAPHY.sizes.sm,
    lineHeight: 18,
  },
  dashboardButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    marginHorizontal: SPACING.lg,
    marginBottom: SPACING.md,
    paddingVertical: SPACING.sm,
    borderRadius: 8,
    gap: SPACING.xs,
  },
  dashboardText: {
    color: '#FFFFFF',
    fontSize: TYPOGRAPHY.sizes.sm,
    fontWeight: TYPOGRAPHY.weights.semibold,
  },
  actionButtons: {
    flexDirection: 'row',
    paddingHorizontal: SPACING.lg,
    marginBottom: SPACING.lg,
    gap: SPACING.sm,
  },
  actionButton: {
    flex: 1,
    paddingVertical: SPACING.sm,
    borderRadius: 8,
    alignItems: 'center',
  },
  actionButtonText: {
    fontSize: TYPOGRAPHY.sizes.sm,
    fontWeight: TYPOGRAPHY.weights.semibold,
  },
  gridPlaceholder: {
    paddingVertical: SPACING.xxl * 2,
    alignItems: 'center',
  },
  gridText: {
    fontSize: TYPOGRAPHY.sizes.base,
    marginTop: SPACING.lg,
  },
});