import React, { useState } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StyleSheet,
  ScrollView,
  Switch,
} from 'react-native';
import { Stack, useRouter } from 'expo-router';
import { Ionicons, MaterialIcons } from '@expo/vector-icons';
import { useSafeAreaInsets } from 'react-native-safe-area-context';
import { useTheme } from '../hooks/useTheme';
import { useAuth } from '../hooks/useAuth';
import { CustomAlert } from '../components/ui/CustomAlert';
import { SPACING, TYPOGRAPHY } from '../constants/theme';

export default function SettingsScreen() {
  const { colors } = useTheme();
  const { logout } = useAuth();
  const insets = useSafeAreaInsets();
  const router = useRouter();

  const [notifications, setNotifications] = useState(true);
  const [privateAccount, setPrivateAccount] = useState(false);
  const [logoutAlert, setLogoutAlert] = useState(false);

  const handleLogout = () => {
    setLogoutAlert(true);
  };

  const confirmLogout = () => {
    logout();
    router.replace('/');
  };

  const SettingItem = ({
    icon,
    title,
    subtitle,
    onPress,
    showArrow = true,
    rightElement,
  }: {
    icon: string;
    title: string;
    subtitle?: string;
    onPress?: () => void;
    showArrow?: boolean;
    rightElement?: React.ReactNode;
  }) => (
    <TouchableOpacity
      style={[styles.settingItem, { borderBottomColor: colors.border }]}
      onPress={onPress}
      disabled={!onPress}
    >
      <View style={styles.settingLeft}>
        <MaterialIcons name={icon as any} size={24} color={colors.text} />
        <View style={styles.settingText}>
          <Text style={[styles.settingTitle, { color: colors.text }]}>{title}</Text>
          {subtitle && (
            <Text style={[styles.settingSubtitle, { color: colors.textSecondary }]}>
              {subtitle}
            </Text>
          )}
        </View>
      </View>
      {rightElement || (showArrow && (
        <Ionicons name="chevron-forward" size={20} color={colors.textSecondary} />
      ))}
    </TouchableOpacity>
  );

  return (
    <>
      <Stack.Screen
        options={{
          headerShown: true,
          headerTitle: 'Settings',
          headerStyle: { backgroundColor: colors.background },
          headerTintColor: colors.text,
          headerShadowVisible: false,
        }}
      />
      <View style={[styles.container, { backgroundColor: colors.background }]}>
        <ScrollView>
          <View style={styles.section}>
            <Text style={[styles.sectionTitle, { color: colors.textSecondary }]}>
              Account
            </Text>
            <SettingItem
              icon="person"
              title="Edit Profile"
              subtitle="Change your profile information"
              onPress={() => {}}
            />
            <SettingItem
              icon="lock"
              title="Privacy"
              subtitle="Control your privacy settings"
              onPress={() => {}}
            />
            <SettingItem
              icon="security"
              title="Security"
              subtitle="Password, two-factor authentication"
              onPress={() => {}}
            />
          </View>

          <View style={styles.section}>
            <Text style={[styles.sectionTitle, { color: colors.textSecondary }]}>
              Preferences
            </Text>
            <SettingItem
              icon="notifications"
              title="Notifications"
              subtitle="Push notifications and emails"
              showArrow={false}
              rightElement={
                <Switch
                  value={notifications}
                  onValueChange={setNotifications}
                  trackColor={{ false: colors.border, true: colors.primary }}
                />
              }
            />
            <SettingItem
              icon="language"
              title="Language"
              subtitle="English"
              onPress={() => {}}
            />
          </View>

          <View style={styles.section}>
            <Text style={[styles.sectionTitle, { color: colors.textSecondary }]}>
              About
            </Text>
            <SettingItem
              icon="help"
              title="Help Center"
              onPress={() => {}}
            />
            <SettingItem
              icon="info"
              title="About"
              onPress={() => {}}
            />
            <SettingItem
              icon="gavel"
              title="Terms & Privacy"
              onPress={() => {}}
            />
          </View>

          <TouchableOpacity
            style={[styles.logoutButton, { backgroundColor: colors.error + '10' }]}
            onPress={handleLogout}
          >
            <MaterialIcons name="logout" size={24} color={colors.error} />
            <Text style={[styles.logoutText, { color: colors.error }]}>
              Log Out
            </Text>
          </TouchableOpacity>
        </ScrollView>

        <CustomAlert
          visible={logoutAlert}
          title="Log Out"
          message="Are you sure you want to log out?"
          buttons={[
            { text: 'Cancel', style: 'cancel', onPress: () => {} },
            { text: 'Log Out', style: 'destructive', onPress: confirmLogout },
          ]}
          onDismiss={() => setLogoutAlert(false)}
        />
      </View>
    </>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
  },
  section: {
    marginTop: SPACING.lg,
  },
  sectionTitle: {
    fontSize: TYPOGRAPHY.sizes.sm,
    fontWeight: TYPOGRAPHY.weights.semibold,
    textTransform: 'uppercase',
    paddingHorizontal: SPACING.lg,
    marginBottom: SPACING.sm,
  },
  settingItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: SPACING.md,
    paddingHorizontal: SPACING.lg,
    borderBottomWidth: 1,
  },
  settingLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  settingText: {
    marginLeft: SPACING.lg,
    flex: 1,
  },
  settingTitle: {
    fontSize: TYPOGRAPHY.sizes.base,
    fontWeight: TYPOGRAPHY.weights.medium,
  },
  settingSubtitle: {
    fontSize: TYPOGRAPHY.sizes.sm,
    marginTop: 2,
  },
  logoutButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    marginHorizontal: SPACING.lg,
    marginVertical: SPACING.xxl,
    paddingVertical: SPACING.md,
    borderRadius: 8,
    gap: SPACING.sm,
  },
  logoutText: {
    fontSize: TYPOGRAPHY.sizes.base,
    fontWeight: TYPOGRAPHY.weights.semibold,
  },
});