import React from 'react';
import { View, TouchableOpacity, Text, StyleSheet, Platform } from 'react-native';
import { Image } from 'expo-image';
import { LinearGradient } from 'expo-linear-gradient';
import { useTheme } from '../hooks/useTheme';
import { SPACING, TYPOGRAPHY } from '../constants/theme';

interface StoryAvatarProps {
  avatar: string;
  username: string;
  isViewed?: boolean;
  onPress: () => void;
}

export function StoryAvatar({
  avatar,
  username,
  isViewed = false,
  onPress,
}: StoryAvatarProps) {
  const { colors } = useTheme();

  return (
    <TouchableOpacity style={styles.container} onPress={onPress}>
      <View style={styles.avatarContainer}>
        {!isViewed ? (
          <LinearGradient
            colors={['#F58529', '#DD2A7B', '#8134AF', '#515BD4']}
            start={{ x: 0, y: 0 }}
            end={{ x: 1, y: 1 }}
            style={styles.gradientRing}
          >
            <View style={[styles.innerRing, { backgroundColor: colors.background }]}>
              <Image source={{ uri: avatar }} style={styles.avatar} />
            </View>
          </LinearGradient>
        ) : (
          <View style={[styles.viewedRing, { borderColor: colors.border }]}>
            <Image source={{ uri: avatar }} style={styles.avatar} />
          </View>
        )}
      </View>
      <Text
        style={[styles.username, { color: colors.text }]}
        numberOfLines={1}
        ellipsizeMode="tail"
      >
        {username}
      </Text>
    </TouchableOpacity>
  );
}

const styles = StyleSheet.create({
  container: {
    alignItems: 'center',
    marginRight: SPACING.md,
    width: 80,
  },
  avatarContainer: {
    marginBottom: SPACING.xs,
  },
  gradientRing: {
    width: 66,
    height: 66,
    borderRadius: 33,
    padding: 2,
  },
  innerRing: {
    width: 62,
    height: 62,
    borderRadius: 31,
    padding: 2,
  },
  viewedRing: {
    width: 66,
    height: 66,
    borderRadius: 33,
    borderWidth: 2,
    padding: 2,
  },
  avatar: {
    width: 58,
    height: 58,
    borderRadius: 29,
  },
  username: {
    fontSize: TYPOGRAPHY.sizes.xs,
    textAlign: 'center',
  },
});