import React, { createContext, useState, ReactNode } from 'react';
import { Post, Comment, SignalType } from '../types';
import { MOCK_POSTS } from '../constants/mockData';

interface PostsContextType {
  posts: Post[];
  toggleLike: (postId: string) => void;
  toggleBookmark: (postId: string) => void;
  addSignal: (postId: string, signal: SignalType) => void;
  addComment: (postId: string, text: string) => void;
  getComments: (postId: string) => Comment[];
}

export const PostsContext = createContext<PostsContextType | undefined>(undefined);

export function PostsProvider({ children }: { children: ReactNode }) {
  const [posts, setPosts] = useState<Post[]>(MOCK_POSTS);
  const [comments, setComments] = useState<Comment[]>([]);

  const toggleLike = (postId: string) => {
    setPosts(prev =>
      prev.map(post =>
        post.id === postId
          ? {
              ...post,
              isLiked: !post.isLiked,
              likesCount: post.isLiked ? post.likesCount - 1 : post.likesCount + 1,
            }
          : post
      )
    );
  };

  const toggleBookmark = (postId: string) => {
    setPosts(prev =>
      prev.map(post =>
        post.id === postId ? { ...post, isBookmarked: !post.isBookmarked } : post
      )
    );
  };

  const addSignal = (postId: string, signal: SignalType) => {
    setPosts(prev =>
      prev.map(post => {
        if (post.id !== postId) return post;

        const hadSignal = post.hasUserSignaled;
        const newYesCount =
          signal === 'yes'
            ? post.yesCount + 1
            : hadSignal === 'yes'
            ? post.yesCount - 1
            : post.yesCount;
        const newNoCount =
          signal === 'no'
            ? post.noCount + 1
            : hadSignal === 'no'
            ? post.noCount - 1
            : post.noCount;

        return {
          ...post,
          hasUserSignaled: hadSignal === signal ? undefined : signal,
          yesCount: newYesCount,
          noCount: newNoCount,
        };
      })
    );
  };

  const addComment = (postId: string, text: string) => {
    const post = posts.find(p => p.id === postId);
    if (!post) return;

    const newComment: Comment = {
      id: `c${Date.now()}`,
      postId,
      userId: 'u1',
      user: post.user,
      text,
      timestamp: new Date(),
      likesCount: 0,
      isLiked: false,
    };

    setComments(prev => [...prev, newComment]);
    setPosts(prev =>
      prev.map(p =>
        p.id === postId ? { ...p, commentsCount: p.commentsCount + 1 } : p
      )
    );
  };

  const getComments = (postId: string) => {
    return comments.filter(c => c.postId === postId);
  };

  return (
    <PostsContext.Provider
      value={{
        posts,
        toggleLike,
        toggleBookmark,
        addSignal,
        addComment,
        getComments,
      }}
    >
      {children}
    </PostsContext.Provider>
  );
}